### R-Script supplied with Pirana to calculate power for Simcyp VBE as a function of subject count
###
### Required:
###   - Reference group {vbe} with predictions file generated by the Simcyp VBE Shiny application (e.g., {vbe}_reference_group.csv)
###   - One or more test group {vbe} with prediction file{s} (e.g., {vbe}_test_group.csv)
###   - Optional PE data file (observed concentrations from Simcyp workspace) for estimating residual variability
###
### Description:
###   This script estimates the empirical power of declaring bioequivalence across a range of subjects per arm.
###   For each model:
###     - Log-normal noise is added to simulated concentrations using either a user-defined or data-estimated sigma
###     - Cmax and AUC are calculated for each subject in both reference and test groups
###     - Subjects are resampled into mini-trials with N subjects per arm (up to user-specified max)
###     - Log-scale Two One-Sided Tests (TOST) are applied on each mini-trial's GMR (Cmax and AUC)
###     - The empirical power is defined as the proportion of trials meeting bioequivalence criteria
###     - Results are saved as individual or combined power-vs-N plots with both raw and smoothed curves
###     - A cached `.Rds` object is saved for downstream Type I error and safe space analyses
###

### <arguments>
###       <n_subj_val label="Subjects per arm" type="text">500</n_subj_val>
###       <resample_with_replacement label="Resample with replacement" type="bool">0</resample_with_replacement>
###       <custom_sigma_val label="Sigma" type="text">.2991788</custom_sigma_val>
###       <use_sigma_obs label="Compute sigma from obs" type="bool">0</use_sigma_obs>
###       <obs_scale_val label="Obs scale" type="text">1</obs_scale_val>
###       <time_filter_val label="Filter time" type="text">8902</time_filter_val>
###       <use_combined label="Combine plots" type="bool">1</use_combined>
### </arguments>

### Arguments supplied from Pirana:
arg <- list (
  n_subj_val = "500",
  resample_with_replacement = "0",
  custom_sigma_val = ".2991788",
  use_sigma_obs = "0",
  obs_scale_val = "1",
  time_filter_val = "8902",
  use_combined = "1"
)


# Setup ----
## Import .vbe and associated files ----
models <- list ( 
  "vbe_1" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_1_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_1_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=4.6357",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "D:/Users/james.craig/VBE/PP3/vbe_1_reference_group.csv",
    test_file       = "",
    type            = "reference",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 4.6357

  ),
  "vbe_2" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_2_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_2_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=1.80",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_2_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 1.8

  ),
  "vbe_3" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_3_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_3_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=2.0",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_3_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 2

  ),
  "vbe_4" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_4_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_4_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=2.6",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_4_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 2.6

  ),
  "vbe_5" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_5_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_5_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=2.7762",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_5_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 2.7762

  ),
  "vbe_6" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_6_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_6_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=3.0",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_6_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 3

  ),
  "vbe_7" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_7_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_7_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=4.6357",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_7_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 4.6357

  ),
  "vbe_8" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_8_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_8_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=4.8375",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_8_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 4.8357

  ),
  "vbe_9" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_9_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_9_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=5.0679",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_9_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 5.0679

  ),
  "vbe_10" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_10_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_10_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=5.5",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_10_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 5.5

  ),
  "vbe_11" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_11_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_11_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.23",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_11_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.23

  ),
  "vbe_12" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_12_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_12_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.45",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_12_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.45

  ),
  "vbe_13" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_13_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_13_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.7",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_13_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.7

  ),
  "vbe_14" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_14_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_14_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.823",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_14_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.8233

  ),
  "vbe_15" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_15_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_15_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=7.1",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_15_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 7.1

  ),
  "vbe_16" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_16_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_16_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=7.5",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_16_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 7.5

  )
)
run_from <- list(software = "pirana", version = "25.7.1")
open_res <- 1

setwd('D:/Users/james.craig/VBE/PP3')

start_time <- Sys.time()
message("Start: ", start_time)

## Process script arguments ----
n_subj_val <- as.numeric(arg$n_subj_val)
use_sigma_obs <- as.logical(as.numeric(arg$use_sigma_obs))
obs_scale_val <- as.numeric(arg$obs_scale_val)
custom_sigma_val  <- as.numeric(arg$custom_sigma_val)
time_filter_val <- as.numeric(arg$time_filter_val)
use_combined <-  as.logical(as.numeric(arg$use_combined))
resample_with_replacement <- as.logical(as.numeric(arg$resample_with_replacement))


## Validation ----
if (length(models) == 0) {
  stop(
    "Results missing for selected `.vbe`, ensure either {vbe}_reference_group.csv or {vbe}_test_group.csv exist in Pirana directory."
  )
}

reference_model <- models[sapply(models, function(x) x$type == "reference")]
test_models <- models[sapply(models, function(x) x$type == "test")]

# Validate reference models
if (length(reference_model) == 0) {
  stop("Validation error: You must select exactly one 'reference' VBE model, but none were selected.")
} else if (length(reference_model) > 1) {
  stop(sprintf("Validation error: You must select exactly one 'reference' VBE model, but %d were selected.", length(reference_model)))
}

# Validate test models
if (length(test_models) == 0) {
  stop("Validation error: At least one 'test' VBE model must be selected, but none were selected.")
}

# Prepare directories ----
if (!file.exists("pirana_reports")) {
  dir.create ("pirana_reports")
}

if (!file.exists("pirana_temp")) {
  dir.create ("pirana_temp")
}

## Import libraries ----
if(Sys.getenv("R_LIB") != "") { .libPaths(c(Sys.getenv("R_LIB"), .libPaths())) }

# Check required packages
req_pkgs <- c("data.table", "Simcyp", "tidyr", "ggplot2")
missing_pkgs <- req_pkgs[!sapply(req_pkgs, requireNamespace, quietly = TRUE)]
if (length(missing_pkgs)) {
  stop("Package(s) missing: ", paste(missing_pkgs, collapse = ", "),
       "\nInstall with: install.packages(c(",
       paste(shQuote(missing_pkgs), collapse = ", "), "))",
       call. = FALSE)
}

library(ggplot2)

## Create Functions ----
get.AUC <- function(times, conc) {
  dx <- diff(times)
  (colSums((conc[-1, ] + conc[-nrow(conc), ]) * dx)) / 2
}

get.Cmax <- function(conc) {
  return(apply(conc, MARGIN=2, FUN=max))
}

myTOST <- function(logX.ref, logX.test, N) {
  mu.ref  <- mean(logX.ref)
  mu.test <- mean(logX.test)

  var.ref  <- var(logX.ref)
  var.test <- var(logX.test)

  logmeans.diff <- mu.test - mu.ref
  logvars.mean  <- (var.ref + var.test) / 2

  delta <- qt(0.95, df = 2*N - 2) * sqrt(logvars.mean * 2 / N)
  CL_lo <- logmeans.diff - delta
  CL_up <- logmeans.diff + delta

  (exp(CL_lo) > 0.8) && (exp(CL_up) < 1.25)
}

# Analysis ----

# Import reference file and prepare data
reference_data <- data.table::fread(reference_model[[1]]$reference_file)

if (use_sigma_obs) {
  message("Computing sigma from observed data (pe_file)...")

  stopifnot(file.exists(pe_file))

  pe_data <- Simcyp::ReadPEData(path = pe_file, scale = obs_scale_val)$Observations
  sim_times <- sort(unique(reference_data$Time))
  subject_ids <- intersect(unique(reference_data$ID), unique(pe_data$SubjectID))

  if (length(subject_ids) == 0) {
    stop("No matching SubjectIDs found between simulated and observed data.")
  }

  all_residuals <- c()

  for (sid in subject_ids) {
    pred <- reference_data[ID == sid, Conc]
    obs <- pe_data[pe_data$SubjectID == sid, ]

    if (nrow(obs) == 0 || length(pred) == 0) next

    pred_interp <- approx(sim_times, pred, obs$Time, rule = 2)$y
    dv_obs <- obs$DV

    if (length(pred_interp) != length(dv_obs)) next

    res <- log(dv_obs) - log(pred_interp)
    all_residuals <- c(all_residuals, res)
  }

  if (length(all_residuals) < 5) {
    warning("Fewer than 5 usable residuals found.")
  }

  sd_residuals <- sd(all_residuals, na.rm = TRUE)
  sigma <- sqrt(log(1 + sd_residuals^2))
  message("Estimated sigma from ",
          length(subject_ids),
          " subject(s): ",
          round(sigma, 4))
} else {
  sigma <- custom_sigma_val
  if (is.na(sigma)) {
    message("No sigma specified. No noise will be added to simulations.")
  } else {
    message("Using user-specified sigma: ", sigma)
  }
}

if (!is.na(sigma)) {
  reference_data[, Conc := rlnorm(.N, meanlog = log(Conc), sdlog = sigma)]
}

reference_data[, ID_prefixed := paste0("RefSubj", ID)]

wide_ref <- data.table::dcast(
  data     = reference_data,
  formula  = Time ~ ID_prefixed,
  value.var= "Conc"
)

# Import test files(s)
model_names <- names(test_models)

if (use_combined) {
  plot_list <- list()
}

for (i in seq(model_names)) {
  model     <- model_names[i]
  message("Processing ", model)

  test_data <- data.table::fread(models[[model]]$test_file)

  if (any(unique(reference_data$Time)  != unique(test_data$Time))) {
    warning("Time mismatch between reference and test groups, skipping ", model)
    next
  }

  # # Add noise
  if (!is.na(sigma)) {
    test_data[, Conc := rlnorm(.N, meanlog = log(Conc), sdlog = sigma)]
  }

  test_data[, ID_prefixed := paste0("TestSubj", ID)]

  wide_test <- data.table::dcast(
    data     = test_data,
    formula  = Time ~ ID_prefixed,
    value.var= "Conc"
  )

  # Ensure N subject req
  if (ncol(wide_test) > ncol(wide_ref)) {
    wide_test <- wide_test[, 1:ncol(wide_ref)]
  }

  wide_ref_keep <- wide_ref[, 1:ncol(wide_test)]
  vbe_data <- cbind(wide_ref_keep, wide_test[, -1, with = FALSE])
  vbe_data <- as.data.frame(vbe_data)

  ##
  ## Keep only the times in the last dosing period
  last_dosing_time <- time_filter_val
  keep <- which(vbe_data[,1] > last_dosing_time)
  vbe_data <- vbe_data[keep,]

  ## Total number of subjects
  N.subj <- ncol(vbe_data) - 1
  ##
  times <- vbe_data$Time
  plot.times <- times / (24 * 7) # in weeks
  N.times <- length(plot.times)
  ##
  ## Prepare computing Cmax and AUC for each subject
  Cmax <- rep(0, N.subj)
  AUC  <- rep(0, N.subj)
  istart   <- 1
  iend     <- N.times
  itime    <- istart:iend
  ##
  ## find Cmax for each subject
  Cmax <- get.Cmax(vbe_data[itime, 1 + (1:N.subj)])
  ##
  ## find AUC for each subject
  AUC  <- get.AUC(plot.times[itime], vbe_data[, 1 + (1:N.subj)])
  ##
  ## Now process increasing chunks of the results
  power <- numeric(n_subj_val)

  logCmax <- log(Cmax) # precompute logs
  logAUC  <- log(AUC)

  half_N_subj <- N.subj / 2
  indices_ref <- sample.int(half_N_subj)
  indices_test <- half_N_subj + sample.int(half_N_subj)
  my_index <- c(indices_ref, indices_test)

  for (k in 2:n_subj_val) {
    N.mtc.v <- floor((length(my_index)) / (2*k))

    BE.yes <- logical(N.mtc.v)

    for (i in seq_len(N.mtc.v)) {
      ref_idx <- sample.int(N.subj / 2, k, replace = resample_with_replacement)
      test_idx <- sample((N.subj / 2 + 1):N.subj, k, replace = resample_with_replacement)

      logCmax_ref <- logCmax[ref_idx]
      logCmax_test <- logCmax[test_idx]
      logAUC_ref <- logAUC[ref_idx]
      logAUC_test <- logAUC[test_idx]
      # Perform TOST tests
      Cmax.yes <- myTOST(logCmax_ref, logCmax_test, k)
      AUC.yes  <- myTOST(logAUC_ref, logAUC_test, k)

      BE.yes[i] <- Cmax.yes && AUC.yes
    }

    power[k] <- mean(BE.yes)
  }

  ## Smooth version of power curve
  smoothed <- supsmu(2:n_subj_val, power[-1])
  ## Residual variance
  residual.var <- var(power[2:n_subj_val] - smoothed$y)

  ## Create pirana vbe cache
  vbe_cache <- list(power = power,
                    n_subj_val = n_subj_val,
                    smoothed = smoothed,
                    residual.var = residual.var
                    )
  saveRDS(vbe_cache,
          file = paste0(
            "pirana_temp/",
            names(reference_model),
            "_",
            model,
            "_power.Rds"
          ))

  ## Create plot
  if (!use_combined) {
    plot.df <- data.frame(
      x        = 2:n_subj_val,
      power    = power[-1],
      smoothed = smoothed$y
    )

    p <- ggplot(plot.df, aes(x = x)) +
      geom_line(aes(y = power), color = "grey", linewidth = 0.5) +
      geom_line(aes(y = smoothed), color = "red", linewidth = 1.2) +
      scale_x_continuous(limits = c(1, n_subj_val)) +
      scale_y_continuous(limits = c(0, 1)) +
      labs(
        title = paste0("Probability of Bioequivalence vs Number of Subjects per Arm: ", model),
        subtitle = paste0(models[[model]]$cqa_name, " = ", models[[model]]$cqa_value),
        x = "Number of subjects per arm",
        y = "Probability of declaring bioequivalence"
      ) +
      theme_minimal()

    fname <-
      paste0("pirana_reports/",
             names(reference_model), "_",
             model,
             "_power_",
             n_subj_val, "_arm.pdf")

    pdf(fname, width=8, height=6)
    print(p)
    dev.off()
    # open created file
    cat (paste("OUTPUT: ", fname, sep=""))
    if (file.exists(fname) && open_res) {
      if (Sys.info()['sysname'] == 'Windows') { shell.exec(paste(getwd(),"/",fname,sep="")) }  # windows
      else if (Sys.info()['sysname'] == 'Darwin') { system(paste ("open ",fname, sep="")) } # mac
      else { system(paste("xdg-open ", fname, sep=""), ignore.stdout=TRUE, ignore.stderr=TRUE, wait=FALSE) } # linux
    }
  } else {
    plot_list[[model]] <- vbe_cache
  }
} # end for fname

if (use_combined) {
  # Create combined plot
  fname <-
    paste0("pirana_reports/",
           names(reference_model), "_",
           paste0(names(test_models),collapse = "_"),
           "_power_",
           n_subj_val, "_arm.pdf")

  combined_all <- purrr::map2_df(
    plot_list,
    test_models,
    function(obj, mod) {
      # Build a per-test data frame
      data.frame(
        x        = 2:obj$n_subj_val,
        power    = obj$power[-1],
        smoothed = obj$smoothed$y,
        model_desc = paste0(mod$cqa_name, " = ", mod$cqa_value)
      )
    }
  )

  combined_long <- combined_all |>
    tidyr::pivot_longer(
      cols = c(power, smoothed),
      names_to = "curve_type",
      values_to = "value"
    )

  message("Creating combined power plot")
  pdf(fname, width = 8, height = 6)

  p <- ggplot(combined_long, aes(x = x, y = value)) +
    geom_line(
      aes(color = model_desc, size = curve_type),
      show.legend = TRUE
    ) +
    scale_size_manual(
      values = c(power = 0.5, smoothed = 1.5),
      guide = "none"  # Turn off legend for size
    ) +
    scale_x_continuous(limits = c(1, n_subj_val)) +
    scale_y_continuous(limits = c(0, 1)) +
    labs(
      title = "Probability of Bioequivalence vs Number of Subjects per Arm",
      x = "Number of subjects per arm",
      y = "Probability of declaring bioequivalence",
      color = "Model"
    ) +
    theme_minimal() +
    theme(legend.position = "right")

  print(p)

  dev.off()

  cat (paste("OUTPUT: ", fname, sep=""))
  if (file.exists(fname) &&
      open_res) {
    if (Sys.info()['sysname'] == 'Windows') {
      shell.exec(paste(getwd(), "/", fname, sep = ""))
    }  # windows
    else if (Sys.info()['sysname'] == 'Darwin') {
      system(paste ("open ", fname, sep = ""))
    } # mac
    else {
      system(
        paste("xdg-open ", fname, sep = ""),
        ignore.stdout = TRUE,
        ignore.stderr = TRUE,
        wait = FALSE
      )
    } # linux
  }
}

end_time <- Sys.time()
message("Complete: ", end_time)
message(capture.output(end_time - start_time))


