### R-Script supplied with Pirana to calculate power thresholds at a fixed trial size
###
### Required: - Reference group predictions file generated from Simcyp VBE Shiny application (e.g., {vbe}_reference.csv)
###           - Test group predictions files generated from Simcyp VBE Shiny application
###           - Power simulation cache files generated previously by `calculate_power_subjects_per_arm.R`
###
### Description: This script computes the empirical power to declare bioequivalence at a fixed trial size across different CQA values (e.g., PSD).
### It also determines the minimum number of subjects per arm required to achieve specific power thresholds (e.g., 80%, 90%) for each test formulation.
### The output includes a table of trial sizes required at given powers and a smoothed plot of power at the user-specified trial size.

### <arguments>
###       <trial_size label="Trial Size" type="text">200</trial_size>
###       <trial_powers label="Powers" type="text">0.05, 0.8, 0.85, 0.9</trial_powers>
### </arguments>

### Arguments supplied from Pirana:
arg <- list (
  trial_size = "200",
  trial_powers = "0.05, 0.8, 0.85, 0.9"
)


# Setup ----
## Import .vbe and associated files ----
models <- list ( 
  "vbe_1" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_1_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_1_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=4.6357",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "D:/Users/james.craig/VBE/PP3/vbe_1_reference_group.csv",
    test_file       = "",
    type            = "reference",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 4.6357

  ),
  "vbe_2" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_2_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_2_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=1.80",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_2_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 1.8

  ),
  "vbe_3" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_3_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_3_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=2.0",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_3_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 2

  ),
  "vbe_4" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_4_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_4_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=2.6",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_4_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 2.6

  ),
  "vbe_5" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_5_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_5_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=2.7762",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_5_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 2.7762

  ),
  "vbe_6" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_6_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_6_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=3.0",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_6_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 3

  ),
  "vbe_7" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_7_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_7_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=4.6357",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_7_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 4.6357

  ),
  "vbe_8" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_8_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_8_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=4.8375",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_8_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 4.8357

  ),
  "vbe_9" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_9_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_9_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=5.0679",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_9_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 5.0679

  ),
  "vbe_10" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_10_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_10_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=5.5",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_10_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 5.5

  ),
  "vbe_11" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_11_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_11_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.23",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_11_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.23

  ),
  "vbe_12" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_12_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_12_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.45",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_12_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.45

  ),
  "vbe_13" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_13_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_13_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.7",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_13_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.7

  ),
  "vbe_14" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_14_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_14_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.823",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_14_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.8233

  ),
  "vbe_15" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_15_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_15_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=7.1",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_15_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 7.1

  ),
  "vbe_16" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_16_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_16_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=7.5",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_16_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 7.5

  )
)
run_from <- list(software = "pirana", version = "25.7.1")
open_res <- 1

setwd('D:/Users/james.craig/VBE/PP3')

start_time <- Sys.time()
message("Start: ", start_time)

## Process script arguments ----
trial_size <- as.numeric(arg$trial_size)
trial_powers <- as.numeric(strsplit(arg$trial_powers, split = ",")[[1]])

## Validation ----
if (length(models) == 0) {
  stop(
    "Results missing for selected `.vbe`, ensure either {vbe}_reference_group.csv or {vbe}_test_group.csv exist in Pirana directory."
  )
}

reference_model <- models[sapply(models, function(x) x$type == "reference")]
test_models <- models[sapply(models, function(x) x$type == "test")]

# Validate reference models
if (length(reference_model) == 0) {
  stop("Validation error: You must select exactly one 'reference' VBE model, but none were selected.")
} else if (length(reference_model) > 1) {
  stop(sprintf("Validation error: You must select exactly one 'reference' VBE model, but %d were selected.", length(reference_model)))
}

# Validate test models
if (length(test_models) == 0) {
  stop("Validation error: At least one 'test' VBE model must be selected, but none were selected.")
}

# Prepare directories ----
if (!file.exists("pirana_reports")) {
  dir.create ("pirana_reports")
}

if (!file.exists("pirana_temp")) {
  dir.create ("pirana_temp")
}

## Import libraries ----
if(Sys.getenv("R_LIB") != "") { .libPaths(c(Sys.getenv("R_LIB"), .libPaths())) }

# Check required packages
req_pkgs <- c("data.table", "Simcyp", "tidyr", "ggplot2")
missing_pkgs <- req_pkgs[!sapply(req_pkgs, requireNamespace, quietly = TRUE)]
if (length(missing_pkgs)) {
  stop("Package(s) missing: ", paste(missing_pkgs, collapse = ", "),
       "\nInstall with: install.packages(c(",
       paste(shQuote(missing_pkgs), collapse = ", "), "))",
       call. = FALSE)
}

library(ggplot2)

## Create Functions ----
# Function to efficiently find nearest match indices for trial size given power
find_trial_size_for_power <- function(smoothed, at.powers) {
  sapply(at.powers, function(pwr) {
    if (pwr < min(smoothed$y) || pwr > max(smoothed$y)) {
      NA
    } else {
      which.min(abs(smoothed$y - pwr))
    }
  })
}

# Function to efficiently find power at given trial sizes
find_power_for_trial_size <- function(smoothed, at.sizes) {
  sapply(at.sizes, function(sz) {
    if (sz < min(smoothed$x) || sz > max(smoothed$x)) {
      NA
    } else {
      smoothed$y[which.min(abs(smoothed$x - sz))]
    }
  })
}

# Get CQA values
descriptions <- sapply(test_models, function(x) paste0(x$cqa_name, " = ", x$cqa_value))
cqa_values <- as.numeric(sub(".*?([0-9]+\\.?[0-9]*).*", "\\1", descriptions))

# Import power values from vbe cache
# Prerequisite: Run calculate_power_subjects_per_arm.R
ref_model <- names(reference_model)
power_cache_list <- list()
for (i in seq_along(test_models)) {
  test_model <- names(test_models)[i]
  power_cache_fname <- file.path("pirana_temp", paste0(ref_model, "_", test_model, "_power.Rds"))
  if (file.exists(power_cache_fname)) {
    power_cache_list[[test_model]] <- readRDS(power_cache_fname)
  } else {
    warning(power_cache_fname, " not found. Run `calculate_power_subjects_per_arm.R` to generate. Skipping ", test_model, ".")
    next
  }
}

# Directly access imported `power_cache_list`
smoothed_data <- lapply(power_cache_list, `[[`, "smoothed")
residual_vars <- sapply(power_cache_list, `[[`, "residual.var")

# Compute trial sizes at specified powers for each test directly from C
trial_sizes_at_pwr <- do.call(rbind, lapply(smoothed_data, find_trial_size_for_power, at.powers = trial_powers))

# Organize trial size results into a clear data.frame
pwr_trial_size_df <- as.data.frame(trial_sizes_at_pwr)
rownames(pwr_trial_size_df) <- cqa_values
colnames(pwr_trial_size_df) <- paste0("Power_", trial_powers)
print(pwr_trial_size_df)

# Compute power at specified trial size
pwr_at_trial_sizes <- sapply(smoothed_data, find_power_for_trial_size, at.sizes = trial_size)

# Create final summarized dataframe clearly
pwr_size_df <- data.frame(
  Radius = cqa_values,
  Power = pwr_at_trial_sizes,
  Residual_var = residual_vars
)

# Plot using ggplot2
power_at_size_plot <- ggplot(pwr_size_df, aes(x = Radius, y = Power)) +
  geom_point(size = 2) +
  geom_smooth(se = FALSE, method = "loess", formula = y ~ x) +
  theme_minimal() +
  labs(
    title = "Power to Declare Bioequivalence Across CQA Values",
    y = paste0("Power at N=", trial_size),
    x = "CQA value"
  )

fname <-
  paste0("pirana_reports/",
         names(reference_model), "_power_trial_size_", trial_size,
         "_", paste0(names(test_models),collapse = "_"),
         ".pdf")


message("Creating power curve plot")
pdf(fname, width = 6, height = 5)

# Display the optimized plot
print(power_at_size_plot)

dev.off()

cat (paste("OUTPUT: ", fname, sep=""))
if (file.exists(fname) &&
    open_res) {
  if (Sys.info()['sysname'] == 'Windows') {
    shell.exec(paste(getwd(), "/", fname, sep = ""))
  }  # windows
  else if (Sys.info()['sysname'] == 'Darwin') {
    system(paste ("open ", fname, sep = ""))
  } # mac
  else {
    system(
      paste("xdg-open ", fname, sep = ""),
      ignore.stdout = TRUE,
      ignore.stderr = TRUE,
      wait = FALSE
    )
  } # linux
}

end_time <- Sys.time()
message("Complete: ", end_time)
message(capture.output(end_time - start_time))

