### R-Script supplied with Pirana to generate time concentration predictions plot for Simcyp VBE
###
### Required: - One or more {vbe} with group predictions file generated from Simcyp VBE Shiny application
###           - Optional PE data file
### Description: Creates individual or mean PRED vs TIME plot using ggplot, observed data points are supplied from PE data file.
###

### <arguments>
###       <use_mean label="Mean predictions" type="bool">0</use_mean>
###       <custom_sigma_val label="Sigma" type="text">.2991788</custom_sigma_val>
###       <use_sigma_obs label="Compute sigma from obs" type="bool">0</use_sigma_obs>
###       <time_filter_val label="Filter time" type="text">8902</time_filter_val>
### </arguments>

### Arguments supplied from Pirana:
arg <- list (
  use_mean = "0",
  custom_sigma_val = ".2991788",
  use_sigma_obs = "0",
  time_filter_val = "8902"
)



# Setup ----
## Import .vbe and associated files ----
models <- list ( 
  "vbe_1" = list ( 
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_1_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_1_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=4.6357",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "D:/Users/james.craig/VBE/PP3/vbe_1_reference_group.csv",
    test_file       = "",
    type            = "reference"

  )
)
run_from <- list(software = "pirana", version = "25.7.1")
open_res <- 1

setwd('D:/Users/james.craig/VBE/PP3')


start_time <- Sys.time()
message("Start: ", start_time)

## Process script arguments ----
use_mean <- as.logical(as.numeric(arg$use_mean))
use_sigma_obs <- as.logical(as.numeric(arg$use_sigma_obs))
custom_sigma_val  <- as.numeric(arg$custom_sigma_val)
time_filter_val <- as.numeric(arg$time_filter_val)

## Validation ----
if (length(models) == 0) {
  stop(
    "Results missing for selected `.vbe`, ensure either {vbe}_reference_group.csv or {vbe}_test_group.csv exist in Pirana directory."
  )
}

## Prepare directories ----
if (!file.exists("pirana_reports")) {
  dir.create ("pirana_reports")
}

if (!file.exists("pirana_temp")) {
  dir.create ("pirana_temp")
}

## Import libraries ----
if(Sys.getenv("R_LIB") != "") { .libPaths(c(Sys.getenv("R_LIB"), .libPaths())) }
# Check required packages
req_pkgs <- c("data.table", "Simcyp", "ggplot2")
missing_pkgs <- req_pkgs[!sapply(req_pkgs, requireNamespace, quietly = TRUE)]
if (length(missing_pkgs)) {
  stop("Package(s) missing: ", paste(missing_pkgs, collapse = ", "),
       "\nInstall with: install.packages(c(",
       paste(shQuote(missing_pkgs), collapse = ", "), "))",
       call. = FALSE)
}

library(ggplot2)


model_names <- names(models)
for (i in seq(model_names)) {
  model     <- model_names[i]
  message("Processing ", model)
  vbe_type <- models[[model]]$type
  if (vbe_type == "reference") {
    conc_data <- data.table::fread(models[[model]]$reference_file)
  } else {
    conc_data <- data.table::fread(models[[model]]$test_file)
  }

  pe_file <- models[[model]]$pe_file
  # Determine sigma to use for log-normal noise
  if (use_sigma_obs) {
    message("Computing sigma from observed data (pe_file)...")
    stopifnot(file.exists(pe_file))
    pe_data <- Simcyp::ReadPEData(path = pe_file, scale = 1)$Observations
    sim_times <- sort(unique(conc_data$Time))
    subject_ids <- intersect(unique(conc_data$ID), unique(pe_data$SubjectID))
    if (length(subject_ids) == 0) {
      stop("No matching SubjectIDs found between simulated and observed data.")
    }

    all_residuals <- c()

    for (sid in subject_ids) {
      pred <- conc_data[ID == sid, Conc]
      obs <- pe_data[pe_data$SubjectID == sid, ]

      if (nrow(obs) == 0 || length(pred) == 0) next

      pred_interp <- approx(sim_times, pred, obs$Time, rule = 2)$y
      dv_obs <- obs$DV

      if (length(pred_interp) != length(dv_obs)) next

      res <- log(dv_obs) - log(pred_interp)
      all_residuals <- c(all_residuals, res)
    }

    if (length(all_residuals) < 5) {
      warning("Fewer than 5 usable residuals found.")
    }

    sd_residuals <- sd(all_residuals, na.rm = TRUE)
    sigma <- sqrt(log(1 + sd_residuals^2))
    message("Estimated sigma from ",
            length(subject_ids),
            " subject(s): ",
            round(sigma, 4))
  } else {
    sigma <- custom_sigma_val
    pe_data <- NULL
    if (is.na(sigma)) {
      message("No sigma specified. No noise will be added to simulations.")
    } else {
      message("Using user-specified sigma: ", sigma)
    }
  }

  if (!is.na(sigma)) {
    conc_data[, Conc := rlnorm(.N, meanlog = log(Conc), sdlog = sigma)]
  }

  if (!is.na(time_filter_val)) {
    conc_data <- conc_data[Time > time_filter_val]
  }

  conc_data$ID <- as.factor(conc_data$ID)

  fname <-
    paste0("pirana_reports/",
           model,
           ifelse(is.na(sigma),
                  "_conc_time_pred",
                  "_conc_time_pred_noise")
    )

  if (use_mean) {
    mean_conc_data <- conc_data[, .(MeanConc = mean(Conc)), by = Time]
    fname <- paste0(fname, "_mean.pdf")
    p <- ggplot() +
      geom_line(data=mean_conc_data, aes(x=Time, y=MeanConc))
  } else {
    if (conc_data[, data.table::uniqueN(ID)] > 50) {
      message("Only the first 50 subjects will be included in resulting plot...")
      keep_ids <- conc_data[, unique(ID)][1:50]
      conc_data <- conc_data[ID %in% keep_ids]
    }
    fname <- paste0(fname, ".pdf")
    p <- ggplot() +
      geom_line(data = conc_data, aes(
        x = Time,
        y = Conc,
        color = ID,
        group = ID
      ))
  }

  if (file.exists(pe_file)) {
    message("Plotting observed data points...")
    if (is.null(pe_data)) {
      pe_data <- Simcyp::ReadPEData(path = pe_file, scale = 1)$Observations
    }
    if (!is.na(time_filter_val)) {
      pe_data <- pe_data[pe_data$Time > time_filter_val, ]
    }
    p <- p + geom_point(data = pe_data, aes(x = Time, y = DV))
  }

  p <- p +
    theme_minimal() +
    labs(title = paste0(
      model,
      ": ",
      vbe_type,
      ifelse(use_mean, " mean", ""),
      " predictions",
      ifelse(is.na(sigma), "", " with noise")
    ))

  pdf(fname, width = 8, height = 6)
  print (p)
  dev.off()
  # open created file
  cat (paste("OUTPUT: ", fname, sep = ""))
  if (file.exists(fname) && open_res) {
    if (Sys.info()['sysname'] == 'Windows') {
      shell.exec(paste(getwd(), "/", fname, sep = ""))
    }  # windows
    else if (Sys.info()['sysname'] == 'Darwin') {
      system(paste ("open ", fname, sep = ""))
    } # mac
    else {
      system(
        paste("xdg-open ", fname, sep = ""),
        ignore.stdout = TRUE,
        ignore.stderr = TRUE,
        wait = FALSE
      )
    } # linux
  }
}

end_time <- Sys.time()
message("Complete: ", end_time)
message(capture.output(end_time - start_time))
