### R-Script supplied with Pirana to perform safe space analysis
###
### Required: - Single reference VBE with reference group simulation in Pirana working directory e.g., {vbe_name}_reference_group.csv
###           - One or more test VBE with test group simulations in in Pirana working directory e.g., {vbe_name}_test_group.csv
###           - A completed CQA sensitivity analysis (CQA_CMax_sensitivity.R), which provides the model fit and GMR-to-CQA mapping
###           - A completed power analysis (calculate_power_subjects_per_arm.R), which provides the power curves for all selected {vbe_name}
###
### Description:
### This script generates two complementary analyses to evaluate the impact of a CQA (e.g., PSD mean radius) on bioequivalence (BE) outcomes:
###
### (1) Maximum Safe Space Region - Computes the CQA range that corresponds to geometric mean ratio (GMR) values between 0.8 and 1.25
### using the selected model from the CQA sensitivity analysis.
###
### (2) Power Curve Plot - Computes and visualizes the probability of declaring BE across a range of CQA values for a fixed number
### of subjects per arm at given trial size. It overlays a smooth power curve and highlights BE-safe (green) and unsafe (red) CQA regions.
### Point sizes represent the variance from GMR estimates at each CQA value.

### <arguments>
###       <use_cqa_sensitivity label="Use CQA values from sensitivity analysis?" type="bool">1</use_cqa_sensitivity>
###       <cqa_gmr_lower_val label="CQA value (lower GMR):" type="text">6.8025</cqa_gmr_lower_val>
###       <cqa_gmr_upper_val label="CQA value (upper GMR):" type="text">2.7762</cqa_gmr_upper_val>
###       <trial_size_val label="Trial Size" type="text">200</trial_size_val>
###       <power_threshold_val label="Power threshold" type="text">0.8</power_threshold_val>
### </arguments>

### Arguments supplied from Pirana:
arg <- list (
  use_cqa_sensitivity = "1",
  cqa_gmr_lower_val = "6.8025",
  cqa_gmr_upper_val = "2.7762",
  trial_size_val = "200",
  power_threshold_val = "0.8"
)


# Setup ----
## Import .vbe and associated files ----
models <- list (
  "vbe_1" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_1_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_1_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=4.6357",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "D:/Users/james.craig/VBE/PP3/vbe_1_reference_group.csv",
    test_file       = "",
    type            = "reference",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 4.6357

  ),
  "vbe_2" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_2_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_2_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=1.80",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_2_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 1.8

  ),
  "vbe_3" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_3_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_3_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=2.0",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_3_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 2

  ),
  "vbe_4" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_4_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_4_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=2.6",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_4_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 2.6

  ),
  "vbe_5" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_5_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_5_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=2.7762",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_5_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 2.7762

  ),
  "vbe_6" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_6_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_6_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=3.0",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_6_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 3

  ),
  "vbe_7" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_7_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_7_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=4.6357",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_7_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 4.6357

  ),
  "vbe_8" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_8_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_8_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=4.8375",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_8_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 4.8357

  ),
  "vbe_9" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_9_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_9_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=5.0679",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_9_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 5.0679

  ),
  "vbe_10" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_10_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_10_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=5.5",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_10_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 5.5

  ),
  "vbe_11" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_11_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_11_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.23",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_11_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.23

  ),
  "vbe_12" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_12_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_12_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.45",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_12_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.45

  ),
  "vbe_13" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_13_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_13_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.7",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_13_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.7

  ),
  "vbe_14" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_14_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_14_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=6.823",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_14_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 6.8233

  ),
  "vbe_15" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_15_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_15_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=7.1",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_15_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 7.1

  ),
  "vbe_16" = list (
    wksz_file       = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_16_PP3M 350mg.wksz",
    pe_file         = "D:/Users/james.craig/VBE/PP3/pirana_temp/vbe_16_magnusson_fig6_panel2_350mg.xml",
    description     = "PSD=7.5",
    working_dir     = "D:/Users/james.craig/VBE/PP3",
    reference_file  = "",
    test_file       = "D:/Users/james.craig/VBE/PP3/vbe_16_test_group.csv",
    type            = "test",
    cqa_name        = "idLogNormalRadiusMean",
    cqa_value       = 7.5

  )
)
run_from <- list(software = "pirana", version = "25.7.1")
open_res <- 1

setwd('D:/Users/james.craig/VBE/PP3')

start_time <- Sys.time()
message("Start: ", start_time)

## Process script arguments ----
trial_size_val <- as.numeric(arg$trial_size_val)
power_threshold_val <- as.numeric(arg$power_threshold_val)
cqa_gmr_lower_val <- as.numeric(arg$cqa_gmr_lower_val)
cqa_gmr_upper_val <- as.numeric(arg$cqa_gmr_upper_val)
use_cqa_sensitivity <- as.logical(as.numeric(arg$use_cqa_sensitivity))


## Validation ----
if (length(models) == 0) {
  stop(
    "Results missing for selected `.vbe`, ensure either {vbe}_reference_group.csv or {vbe}_test_group.csv exist in Pirana directory."
  )
}

reference_model <- models[sapply(models, function(x) x$type == "reference")]
test_models <- models[sapply(models, function(x) x$type == "test")]

# Validate reference models
if (length(reference_model) == 0) {
  stop("Validation error: You must select exactly one 'reference' VBE model, but none were selected.")
} else if (length(reference_model) > 1) {
  stop(sprintf("Validation error: You must select exactly one 'reference' VBE model, but %d were selected.", length(reference_model)))
}

# Validate test models
if (length(test_models) == 0) {
  stop("Validation error: At least one 'test' VBE model must be selected, but none were selected.")
}

## Prepare directories ----
if (!file.exists("pirana_reports")) {
  dir.create ("pirana_reports")
}

if (!file.exists("pirana_temp")) {
  dir.create ("pirana_temp")
}

## Import libraries ----
if(Sys.getenv("R_LIB") != "") { .libPaths(c(Sys.getenv("R_LIB"), .libPaths())) }
# Check required packages
req_pkgs <- c("data.table", "Simcyp", "dplyr", "ggplot2")
missing_pkgs <- req_pkgs[!sapply(req_pkgs, requireNamespace, quietly = TRUE)]
if (length(missing_pkgs)) {
  stop("Package(s) missing: ", paste(missing_pkgs, collapse = ", "),
       "\nInstall with: install.packages(c(",
       paste(shQuote(missing_pkgs), collapse = ", "), "))",
       call. = FALSE)
}

library(ggplot2)

find_power_for_trial_size <- function(smoothed, at.sizes) {
  sapply(at.sizes, function(sz) {
    if (sz < min(smoothed$x) || sz > max(smoothed$x)) {
      NA
    } else {
      smoothed$y[which.min(abs(smoothed$x - sz))]
    }
  })
}

# Analysis ----

# Check if CQA metrics exist
fname <-
  paste0("pirana_temp/",
         names(reference_model), "_CQA_metrics_",
         paste0(names(test_models),collapse = "_"),
         ".Rds")

if (file.exists(fname)) {
  CQA_metrics <- readRDS(fname)
} else {
  stop(fname, "\n  Not found. You must first run `CQA_CMax_sensitivity.R` with selected VBE.")
}

# Generate GMR predictions based on the model form
# radius: range of x values on natural scale, supplied by user or automatically imported from CQA sensitivity assessment
# model: selected lm object imported from CQA sensitivity based on lowest sum of absolute residuals
if (use_cqa_sensitivity) {
  cqa_gmr_lower_val <- CQA_metrics$cqa_gmr_lower_val
  cqa_gmr_upper_val <- CQA_metrics$cqa_gmr_upper_val
}

cqa_x_lower <- min(cqa_gmr_lower_val, cqa_gmr_upper_val)
cqa_x_upper <- max(cqa_gmr_lower_val, cqa_gmr_upper_val)

if (any(is.na(c(cqa_x_lower, cqa_x_upper)))) {
  stop("Check CQA values provided")
}

radius <- seq(cqa_x_lower * 0.9, cqa_x_upper * 1.1, length.out = 100)
model <- CQA_metrics$model
fmla <- formula(model)
lhs <- deparse(fmla[[2]])  # e.g., y, exp(y)
rhs <- deparse(fmla[[3]])  # e.g., x, log(x)

# Predict GMR values from the model across radius
GMR <- dplyr::case_when(
  lhs == "exp(y)" & rhs == "log(x)" ~ predict(model, newdata = data.frame(x = radius)),
  lhs == "exp(y)" & rhs == "x"      ~ predict(model, newdata = data.frame(x = radius)),
  lhs == "y"      & rhs == "log(x)" ~ exp(predict(model, newdata = data.frame(x = radius))),
  lhs == "y"      & rhs == "x"      ~ exp(predict(model, newdata = data.frame(x = radius))),
  TRUE ~ NA_real_  # fallback
)

safe_df <- data.frame(radius, GMR)

# Create plot ----
fname <-
  paste0("pirana_reports/",
         names(reference_model), "_safe_space_max_region_",
         paste0(names(test_models),collapse = "_"),
         ".pdf")

pdf(fname, width = 8, height = 6)
p <- ggplot(safe_df, aes(x = radius, y = GMR)) +
  geom_rect(
    data = NULL, xmin = cqa_x_lower, xmax = cqa_x_upper, ymin = 0.8, ymax = 1.25,
    fill = "lightgreen", alpha = 0.6, inherit.aes = FALSE
  ) +
  geom_line(color = "red", linewidth = 1.2) +
  coord_cartesian(ylim = c(0.7, 1.35)) +
  labs(
    title = "True Bioequivalance Region",
    x = expression(paste("PSD mean radius (", mu, "m)")),
    y = expression(paste(C[max], " geometric mean ratio"))
  ) +
  theme_minimal()

print(p)

dev.off()

cat (paste("OUTPUT: ", fname, sep=""))

if (file.exists(fname) &&
    open_res) {
  if (Sys.info()['sysname'] == 'Windows') {
    shell.exec(paste(getwd(), "/", fname, sep = ""))
  }  # windows
  else if (Sys.info()['sysname'] == 'Darwin') {
    system(paste ("open ", fname, sep = ""))
  } # mac
  else {
    system(
      paste("xdg-open ", fname, sep = ""),
      ignore.stdout = TRUE,
      ignore.stderr = TRUE,
      wait = FALSE
    )
  } # linux
}

# Analysis ----
# === Import and preprocess power data ===
power_cache_list <- list()
for (i in seq_along(test_models)) {
  test_model <- names(test_models)[i]
  power_cache_fname <- file.path(
    "pirana_temp",
    paste0(names(reference_model), "_", test_model, "_power.Rds")
  )
  if (file.exists(power_cache_fname)) {
    power_cache_list[[test_model]] <- readRDS(power_cache_fname)
  } else {
    warning(
      power_cache_fname,
      " not found. Run `calculate_power_subjects_per_arm.R` to generate. Skipping ",
      test_model, "."
    )
    next
  }
}

smoothed_data  <- lapply(power_cache_list, `[[`, "smoothed")
residual_vars  <- sapply(power_cache_list, `[[`, "residual.var")

# === Get CQA and Power Data ===
# Get CQA values
descriptions <- sapply(test_models[names(power_cache_list)], function(x) paste0(x$cqa_name, " = ", x$cqa_value))
test_cqa_values     <- as.numeric(sub(".*?([0-9]+\\.?[0-9]*).*", "\\1", descriptions))
ref_cqa_value       <- as.numeric(reference_model[[1]]$cqa_value)

n_subj_val <- unique(sapply(power_cache_list, `[[`, "n_subj_val"))
if (length(n_subj_val) > 1) stop("Different number of subjects used.")

pwr_at_trial_sizes <- sapply(
  smoothed_data,
  find_power_for_trial_size,
  at.sizes = trial_size_val
)

pwr_size_df <- data.frame(
  Radius       = test_cqa_values,
  Power        = pwr_at_trial_sizes,
  Residual_var = residual_vars
)

# === Identify safe space region based on empirical power threshold ===
# find contiguous region where smooth_df >= threshold
# first build smooth_df
x1   <- seq(min(pwr_size_df$Radius), ref_cqa_value, length.out = 500)
mysd1 <- 0.79
uu1  <- dnorm(x1, ref_cqa_value, mysd1)
y1   <- uu1 * max(pwr_size_df$Power) / max(uu1)

x2   <- seq(ref_cqa_value, max(pwr_size_df$Radius), length.out = 500)
mysd2 <- 0.94
uu2  <- dnorm(x2, ref_cqa_value, mysd2)
hinge <- tail(y1, 1)
y2   <- uu2 * hinge / max(uu2)

smooth_df <- rbind(
  data.frame(x = x1, y = y1),
  data.frame(x = x2, y = y2)
)
smooth_df <- smooth_df[order(smooth_df$x), ]

above     <- smooth_df$y >= power_threshold_val
cross_idx <- which(above)
lower_bound <- if (length(cross_idx)) smooth_df$x[min(cross_idx)] else NA
upper_bound <- if (length(cross_idx)) smooth_df$x[max(cross_idx)] else NA

# === Prepare plot data ===
plot_df <- pwr_size_df
plot_df$w   <- CQA_metrics$metric_df_ratios$CmaxR_logvars
var_scaled   <- sqrt(plot_df$w)
var_scaled   <- (var_scaled - min(var_scaled)) /
  (max(var_scaled) - min(var_scaled))
plot_df$cex  <- 1 + var_scaled * 3

# === Define colored regions ===
zone_df <- data.frame(
  xmin = c(
    min(plot_df$Radius),
    cqa_x_lower,
    lower_bound,
    upper_bound,
    cqa_x_upper
  ),
  xmax = c(
    cqa_x_lower,
    lower_bound,
    upper_bound,
    cqa_x_upper,
    max(plot_df$Radius)
  ),
  fill = factor(
    c("outside","model","safe","model","outside"),
    levels = c("outside","model","safe")
  )
)
zone_colors <- c(
  outside = "mistyrose",
  model   = "lightyellow",
  safe    = "lightgreen"
)

lower_y <- approx(smooth_df$x, smooth_df$y, xout = lower_bound)$y
upper_y <- approx(smooth_df$x, smooth_df$y, xout = upper_bound)$y

# === Create plot ===
fname <- file.path(
  "pirana_reports",
  paste0(
    names(reference_model),
    "_safe_space_curve_",
    paste0(names(test_models), collapse = "_"),
    ".pdf"
  )
)
pdf(fname, width = 8, height = 6)

p <- ggplot(plot_df, aes(x = Radius, y = Power)) +
  # colored bands
  geom_rect(
    data = zone_df,
    aes(xmin = xmin, xmax = xmax, ymin = -Inf, ymax = Inf, fill = fill),
    inherit.aes = FALSE, alpha = 0.5
  ) +
  scale_fill_manual(values = zone_colors, guide = "none") +

  # original model-based CQA bounds
  geom_vline(
    xintercept = c(cqa_x_lower, cqa_x_upper),
    color = "red", linetype = "solid"
  ) +

  # empirical power-based bounds
  geom_vline(
    xintercept = c(lower_bound, upper_bound),
    color = "darkgreen", linetype = "dashed"
  ) +

  # points & smooth curve
  geom_point(aes(size = cex), shape = 1, stroke = 0.5) +
  geom_line(
    data = smooth_df,
    aes(x = x, y = y),
    color = "red", linewidth = 1.2
  ) +

  # power threshold line
  geom_hline(yintercept = power_threshold_val, linetype = "dotted") +
  # scales & labels
  scale_x_log10() +
  scale_y_continuous(limits = c(0,1)) +
  scale_size_identity() +
  labs(
    title = paste0("Safe Space Region Based on Power >= ", power_threshold_val,
                   " (N=", n_subj_val, ")"),
    subtitle = paste0("CQA Range (Power >= ", power_threshold_val, "): [",round(lower_bound, 4), " - ", round(upper_bound, 4), "]"),
    x     = "CQA Value",
    y     = paste0("Power with ", n_subj_val, " subjects per arm")
  ) +
  theme_minimal()

print(p)
dev.off()

cat (paste("OUTPUT: ", fname, sep=""))

if (file.exists(fname) &&
    open_res) {
  if (Sys.info()['sysname'] == 'Windows') {
    shell.exec(paste(getwd(), "/", fname, sep = ""))
  }  # windows
  else if (Sys.info()['sysname'] == 'Darwin') {
    system(paste ("open ", fname, sep = ""))
  } # mac
  else {
    system(
      paste("xdg-open ", fname, sep = ""),
      ignore.stdout = TRUE,
      ignore.stderr = TRUE,
      wait = FALSE
    )
  } # linux
}

end_time <- Sys.time()
message("Complete: ", end_time)
message(capture.output(end_time - start_time))

